﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using Mdws2ORM;
using Mdws2ORM.Core;
using Mdws2ORM.Maps.BaseFields;

namespace BMS.VistaIntegration.Via.Commands
{
    public abstract class BaseListCommand<T> : BaseCommand<T> where T : class, IEntity
    {
        private bool hasFileNum;

        protected BaseListCommand(ViaVistAQuery query, bool hasFileNum = false)
            : base(query)
        {
            this.hasFileNum = hasFileNum;
            this.HasMore = true;
        }

        public int? MaxCount
        {
            get;
            set;
        }

        public string From
        {
            get;
            set;
        }

        public string Last
        {
            get;
            protected set;
        }

        public bool HasMore
        {
            get;
            protected set;
        }

        protected virtual bool IsFiltered
        {
            get
            {
                return false;
            }
        }

        protected override List<T> GetEntitiesFromData(ViaVistASession session)
        {
            return this.VistAQuery.EntitySetCache.RunThroghCache<T>(() => ProcessDictionaries(session).Union().Distinct(EqualityComparer).ToSortedList(BaseCommand<T>.Comparer));
        }

        private IEnumerable<IEnumerable<T>> ProcessDictionaries(ViaVistASession session)
        {
            var dictionaries = this.GetSectionsFromVia(session);

            foreach (var dictionaryEntry in dictionaries)
            {
                var dictionary = dictionaryEntry.Value;

                while (dictionary != null && dictionary.ContainsKey("Misc"))
                {
                    this.From = this.GetNext(dictionary["Misc"]);
                    this.MergeDictionaries(dictionaries, this.GetSectionsFromVia(session));
                }

                yield return this.ProcessSingleDictionary(session, dictionaryEntry.Key, dictionary);
            }
        }

        private void MergeDictionaries(Dictionary<string, Dictionary<string, List<string>>> dictionaries, Dictionary<string, Dictionary<string, List<string>>> newDictionaries)
        {
            foreach (var key in dictionaries.Keys)
            {
                var dictionary = dictionaries[key];
                if (newDictionaries.ContainsKey(key))
                {
                    var newDictionary = newDictionaries[key];
                    if (newDictionary.ContainsKey("Data"))
                    {
                        dictionary["Data"].AddRange(newDictionary["Data"]);
                    }
                    dictionary.Remove("Misc");
                    if (newDictionary.ContainsKey("Misc"))
                    {
                        dictionary.Add("Misc", newDictionary["Misc"]);
                    }
                }
            }
        }

        protected virtual IEnumerable<T> ProcessSingleDictionary(ViaVistASession session, string key, Dictionary<string, List<string>> dictionary)
        {
            var lines = dictionary.ContainsKey("Data") ? (IList<string>)dictionary["Data"] : new string[0];
            this.Logger.LogFormat(BmsLogger.Level.Verbose, "VIA returned total of {0} record(s).", lines.Count);

            var queryParam = this.EntityMap.ListQueryParam;
            var entries = this.ProcessLines(queryParam, lines, this.EntityMap.AllFields, this.hasFileNum);
            this.HasMore = false;

            this.PrepareCache(session.VistASite, entries);

            return (from entry in entries
                    select this.GetEntity(entry, EntityMap, ""));
        }

        protected override IEnumerable<object> GetCriteria()
        {
            yield return this.From;
            yield return this.MaxCount;
        }

        protected virtual string TranslateValue(string fieldIen, string value)
        {
            return value;
        }

        private string GetNext(List<string> miscSection)
        {
            var next = string.Empty;
            var moreLine = miscSection.SingleOrDefault(x => x.StartsWith("MORE^", StringComparison.Ordinal));
            if (!string.IsNullOrEmpty(moreLine))
            {
                next = moreLine.Split('^')[1];
            }
            else
            {
                next = null;
            }
            return next;
        }

        private IList<Entry> ProcessLines(QueryParam queryParam, IList<string> lines, IList<BaseFieldMap<T>> allFields, bool hasFileNum)
        {
            var start = hasFileNum ? 1 : 0;

            var result = new Entry[lines.Count];
            for (int i = 0; i < lines.Count; i++)
            {
                var values = lines[i].Split('^');
                var fields = new Entry.Field[allFields.Count];
                var ien = values[start];

                for (var j = 0; j < Math.Min(allFields.Count, values.Length - 1); j++)
                {
                    fields[j] = new Entry.Field(allFields[j].Field, this.TranslateValue(allFields[j].Field, values[j + start + 1]));
                }

                result[i] = new Entry(queryParam.File, ien, fields);
            }

            return result;
        }
    }
}
